<?php
session_start();
include 'db_connect.php';

// Security: Prevent SQL Injection (use prepared statements)
function sanitize_input($data) {
    global $conn;
    return mysqli_real_escape_string($conn, trim($data));
}

// Check if form was submitted
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    header("Location: add_location.php?error=Invalid request method");
    exit();
}

// Get and sanitize inputs
$location_code = sanitize_input($_POST['location_code'] ?? '');
$location_name = sanitize_input($_POST['location_name'] ?? '');
$state_offices = sanitize_input($_POST['state_offices'] ?? '');

// Validation
$errors = [];

if (empty($location_code)) {
    $errors[] = "Location Code is required";
} elseif (strlen($location_code) < 2) {
    $errors[] = "Location Code must be at least 2 characters";
}

if (empty($location_name)) {
    $errors[] = "Location Name is required";
} elseif (strlen($location_name) < 3) {
    $errors[] = "Location Name must be at least 3 characters";
}

if (empty($state_offices)) {
    $errors[] = "State Offices are required";
}

// Check if location code already exists
$check_sql = "SELECT id FROM location_master WHERE location_code = '$location_code'";
$check_result = $conn->query($check_sql);
if ($check_result && $check_result->num_rows > 0) {
    $errors[] = "Location Code already exists. Please use a different code.";
}

// If there are errors, redirect back with error message
if (!empty($errors)) {
    $error_message = implode("<br>", $errors);
    header("Location: add_location.php?error=" . urlencode($error_message));
    exit();
}

// Insert into database using prepared statement for security
$sql = "INSERT INTO location_master (location_code, location_name, state_offices) VALUES (?, ?, ?)";
$stmt = $conn->prepare($sql);

if ($stmt) {
    $stmt->bind_param("sss", $location_code, $location_name, $state_offices);
    
    if ($stmt->execute()) {
        $new_location_id = $stmt->insert_id;
        $stmt->close();
        
        // Success - show beautiful success page
        show_success_page($location_code, $location_name);
    } else {
        // Database error
        $stmt->close();
        header("Location: add_location.php?error=" . urlencode("Database error: " . $conn->error));
        exit();
    }
} else {
    // Prepare statement failed
    header("Location: add_location.php?error=" . urlencode("System error. Please try again."));
    exit();
}

$conn->close();

function show_success_page($location_code, $location_name) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Success - Location Added | #JRM BALASORE</title>
        <!-- Bootstrap 5 -->
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
        <!-- Font Awesome -->
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        <style>
            :root {
                --primary-color: #4361ee;
                --primary-dark: #3a56d4;
                --secondary-color: #3a0ca3;
                --success-color: #10b981;
                --light-color: #f8f9fa;
                --dark-color: #212529;
                --card-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
                --card-radius: 16px;
                --transition: all 0.3s ease;
            }
            
            body {
                background: linear-gradient(135deg, #f5f7ff 0%, #eef2ff 100%);
                min-height: 100vh;
                font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 20px;
            }
            
            .success-container {
                max-width: 600px;
                width: 100%;
                animation: fadeIn 0.6s ease-out;
            }
            
            @keyframes fadeIn {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            .success-card {
                background: white;
                border-radius: var(--card-radius);
                box-shadow: var(--card-shadow);
                border: none;
                padding: 50px 40px;
                text-align: center;
                position: relative;
                overflow: hidden;
            }
            
            .success-card::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                height: 5px;
                background: linear-gradient(135deg, var(--success-color) 0%, #34d399 100%);
            }
            
            .success-icon {
                width: 100px;
                height: 100px;
                background: linear-gradient(135deg, var(--success-color) 0%, #34d399 100%);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                margin: 0 auto 30px;
                box-shadow: 0 10px 25px rgba(16, 185, 129, 0.3);
                animation: scaleIn 0.5s ease-out 0.2s both;
            }
            
            @keyframes scaleIn {
                from {
                    transform: scale(0);
                }
                to {
                    transform: scale(1);
                }
            }
            
            .success-icon i {
                font-size: 40px;
                color: white;
            }
            
            .success-title {
                font-size: 2.2rem;
                font-weight: 800;
                color: var(--dark-color);
                margin-bottom: 15px;
            }
            
            .success-message {
                font-size: 1.1rem;
                color: #64748b;
                margin-bottom: 30px;
                line-height: 1.6;
            }
            
            .location-details {
                background: #f8fafc;
                border-radius: 12px;
                padding: 25px;
                margin: 30px 0;
                border: 1px solid #e2e8f0;
                text-align: left;
            }
            
            .detail-item {
                display: flex;
                align-items: flex-start;
                margin-bottom: 15px;
                padding-bottom: 15px;
                border-bottom: 1px solid #e2e8f0;
            }
            
            .detail-item:last-child {
                margin-bottom: 0;
                padding-bottom: 0;
                border-bottom: none;
            }
            
            .detail-icon {
                color: var(--primary-color);
                font-size: 1.2rem;
                width: 30px;
                margin-right: 15px;
                margin-top: 2px;
            }
            
            .detail-content h5 {
                font-size: 0.9rem;
                font-weight: 600;
                color: #64748b;
                text-transform: uppercase;
                letter-spacing: 0.5px;
                margin-bottom: 5px;
            }
            
            .detail-content p {
                font-size: 1.1rem;
                font-weight: 500;
                color: var(--dark-color);
                margin: 0;
            }
            
            .action-buttons {
                display: flex;
                gap: 15px;
                justify-content: center;
                flex-wrap: wrap;
                margin-top: 40px;
            }
            
            .btn-success-custom {
                background: linear-gradient(135deg, var(--success-color) 0%, #34d399 100%);
                border: none;
                border-radius: 12px;
                padding: 14px 30px;
                font-weight: 600;
                color: white;
                transition: var(--transition);
                box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                gap: 10px;
            }
            
            .btn-success-custom:hover {
                transform: translateY(-3px);
                box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
                color: white;
            }
            
            .btn-primary-custom {
                background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
                border: none;
                border-radius: 12px;
                padding: 14px 30px;
                font-weight: 600;
                color: white;
                transition: var(--transition);
                box-shadow: 0 4px 15px rgba(67, 97, 238, 0.3);
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                gap: 10px;
            }
            
            .btn-primary-custom:hover {
                transform: translateY(-3px);
                box-shadow: 0 6px 20px rgba(67, 97, 238, 0.4);
                color: white;
            }
            
            .btn-outline-custom {
                background: transparent;
                border: 2px solid #e2e8f0;
                border-radius: 12px;
                padding: 14px 30px;
                font-weight: 600;
                color: #64748b;
                transition: var(--transition);
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                gap: 10px;
            }
            
            .btn-outline-custom:hover {
                background: #f8fafc;
                border-color: var(--primary-color);
                color: var(--primary-color);
                transform: translateY(-2px);
            }
            
            .confetti {
                position: absolute;
                width: 10px;
                height: 10px;
                background: var(--success-color);
                border-radius: 50%;
                opacity: 0.7;
                animation: confetti-fall 3s linear forwards;
            }
            
            @keyframes confetti-fall {
                0% {
                    transform: translateY(-100px) rotate(0deg);
                    opacity: 1;
                }
                100% {
                    transform: translateY(500px) rotate(360deg);
                    opacity: 0;
                }
            }
            
            @media (max-width: 768px) {
                .success-card {
                    padding: 40px 25px;
                }
                
                .success-title {
                    font-size: 1.8rem;
                }
                
                .action-buttons {
                    flex-direction: column;
                }
                
                .btn-success-custom,
                .btn-primary-custom,
                .btn-outline-custom {
                    width: 100%;
                    justify-content: center;
                }
            }
            
            @media (max-width: 576px) {
                body {
                    padding: 15px;
                }
                
                .success-card {
                    padding: 30px 20px;
                }
                
                .success-icon {
                    width: 80px;
                    height: 80px;
                }
                
                .success-icon i {
                    font-size: 32px;
                }
                
                .success-title {
                    font-size: 1.6rem;
                }
                
                .location-details {
                    padding: 20px 15px;
                }
            }
        </style>
    </head>
    <body>
        <div class="success-container">
            <div class="success-card">
                <!-- Confetti Animation -->
                <div id="confetti-container"></div>
                
                <div class="success-icon">
                    <i class="fas fa-check"></i>
                </div>
                
                <h1 class="success-title">Location Added Successfully!</h1>
                
                <p class="success-message">
                    The new location has been successfully added to the system.
                    You can now add parties to this location or create more locations.
                </p>
                
                <!-- Location Details -->
                <div class="location-details">
                    <div class="detail-item">
                        <div class="detail-icon">
                            <i class="fas fa-hashtag"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Location Code</h5>
                            <p><?php echo htmlspecialchars($location_code); ?></p>
                        </div>
                    </div>
                    
                    <div class="detail-item">
                        <div class="detail-icon">
                            <i class="fas fa-building"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Location Name</h5>
                            <p><?php echo htmlspecialchars($location_name); ?></p>
                        </div>
                    </div>
                    
                    <div class="detail-item">
                        <div class="detail-icon">
                            <i class="fas fa-calendar-check"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Date Added</h5>
                            <p><?php echo date('F j, Y \a\t g:i A'); ?></p>
                        </div>
                    </div>
                </div>
                
                <!-- Action Buttons -->
                <div class="action-buttons">
                    <a href="add_location.php" class="btn-primary-custom">
                        <i class="fas fa-plus-circle"></i>Add Another Location
                    </a>
                    
                    <a href="add_party.php" class="btn-success-custom">
                        <i class="fas fa-user-friends"></i>Add Party to Location
                    </a>
                    
                    <a href="index.php" class="btn-outline-custom">
                        <i class="fas fa-home"></i>Back to Dashboard
                    </a>
                </div>
            </div>
        </div>
        
        <!-- Bootstrap JS -->
        <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
        
        <script>
        // Create confetti animation
        function createConfetti() {
            const container = document.getElementById('confetti-container');
            const colors = ['#10b981', '#34d399', '#22c55e', '#16a34a'];
            
            for (let i = 0; i < 50; i++) {
                const confetti = document.createElement('div');
                confetti.className = 'confetti';
                
                // Random properties
                const size = Math.random() * 10 + 5;
                const left = Math.random() * 100;
                const color = colors[Math.floor(Math.random() * colors.length)];
                const duration = Math.random() * 2 + 2;
                const delay = Math.random() * 2;
                
                // Apply styles
                confetti.style.width = `${size}px`;
                confetti.style.height = `${size}px`;
                confetti.style.backgroundColor = color;
                confetti.style.left = `${left}%`;
                confetti.style.animationDuration = `${duration}s`;
                confetti.style.animationDelay = `${delay}s`;
                
                container.appendChild(confetti);
                
                // Remove confetti after animation completes
                setTimeout(() => {
                    if (confetti.parentNode) {
                        confetti.remove();
                    }
                }, (duration + delay) * 1000);
            }
        }
        
        // Create confetti on page load
        document.addEventListener('DOMContentLoaded', () => {
            createConfetti();
            
            // Create more confetti every 3 seconds
            setInterval(() => {
                createConfetti();
            }, 3000);
        });
        
        // Auto-redirect after 30 seconds (optional)
        setTimeout(() => {
            if (confirm('Would you like to return to the dashboard?')) {
                window.location.href = 'index.php';
            }
        }, 30000);
        </script>
    </body>
    </html>
    <?php
}
?>