<?php
// save_party.php

include 'db_connect.php';

// Basic input sanitization (you should use prepared statements in production)
$party_no     = $_POST['party_no'];
$party_name   = $_POST['party_name'];
$location_id  = $_POST['location_ids'];
$longitude    = $_POST['longitude'];
$latitude     = $_POST['latitude'];
$distance     = $_POST['distance'];

// Validate required fields
if (empty($party_no) || empty($party_name) || $location_id <= 0 || $longitude == 0 || $latitude == 0) {
    die("Error: All required fields must be filled.");
}

// Handle file upload
$upload_path = '';
if (isset($_FILES['upload_file']) && $_FILES['upload_file']['error'] === UPLOAD_ERR_OK) {
    $file_ext = strtolower(pathinfo($_FILES['upload_file']['name'], PATHINFO_EXTENSION));
    
    if ($file_ext !== 'pdf') {
        die("Error: Only PDF files are allowed.");
    }
    
    $upload_dir = 'uploads/';
    if (!is_dir($upload_dir)) {
        mkdir($upload_dir, 0755, true);
    }
    
    $new_filename = $party_no . '.pdf';
    $upload_path  = $upload_dir . $new_filename;
    
    if (!move_uploaded_file($_FILES['upload_file']['tmp_name'], $upload_path)) {
        die("Error: Failed to upload file.");
    }
} else {
    die("Error: PDF file is required.");
}

// Insert into database (using prepared statement - recommended)
$stmt = $conn->prepare("
    INSERT INTO party 
    (party_no, party_name, location_id, longitude, latitude, distance, upload_file) 
    VALUES (?, ?, ?, ?, ?, ?, ?)
");

$stmt->bind_param(
    "ssiddds",
    $party_no,
    $party_name,
    $location_id,
    $longitude,
    $latitude,
    $distance,
    $upload_path
);

if ($stmt->execute()) {
    // Get the location name for display
    $location_query = $conn->prepare("SELECT location_name FROM location_master WHERE id = ?");
    $location_query->bind_param("i", $location_id);
    $location_query->execute();
    $location_result = $location_query->get_result();
    $location_row = $location_result->fetch_assoc();
    $location_name = $location_row ? $location_row['location_name'] : 'Unknown Location';
    $location_query->close();
    
    $stmt->close();
    $conn->close();
    
    // Show beautiful success page
    show_success_page($party_no, $party_name, $location_name, $longitude, $latitude, $distance, $upload_path);
} else {
    $stmt->close();
    $conn->close();
    die("Database error: Unable to save party information.");
}

function show_success_page($party_no, $party_name, $location_name, $longitude, $latitude, $distance, $upload_path) {
    ?>
    <!DOCTYPE html>
    <html lang="en">
    <head>
        <meta charset="UTF-8">
        <meta name="viewport" content="width=device-width, initial-scale=1.0">
        <title>Success - Party Added | #JRM BALSORE</title>
        <!-- Bootstrap 5 -->
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
        <!-- Font Awesome -->
        <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
        <style>
            :root {
                --primary-color: #4361ee;
                --primary-dark: #3a56d4;
                --secondary-color: #3a0ca3;
                --success-color: #10b981;
                --accent-color: #4cc9f0;
                --light-color: #f8f9fa;
                --dark-color: #212529;
                --card-shadow: 0 10px 30px rgba(0, 0, 0, 0.08);
                --card-radius: 16px;
                --transition: all 0.3s ease;
            }
            
            body {
                background: linear-gradient(135deg, #f5f7ff 0%, #eef2ff 100%);
                min-height: 100vh;
                font-family: 'Segoe UI', system-ui, -apple-system, sans-serif;
                display: flex;
                align-items: center;
                justify-content: center;
                padding: 20px;
                margin: 0;
            }
            
            .success-container {
                max-width: 800px;
                width: 100%;
                animation: fadeIn 0.6s ease-out;
            }
            
            @keyframes fadeIn {
                from {
                    opacity: 0;
                    transform: translateY(30px);
                }
                to {
                    opacity: 1;
                    transform: translateY(0);
                }
            }
            
            .success-card {
                background: white;
                border-radius: var(--card-radius);
                box-shadow: var(--card-shadow);
                border: none;
                padding: 50px 40px;
                text-align: center;
                position: relative;
                overflow: hidden;
            }
            
            .success-card::before {
                content: '';
                position: absolute;
                top: 0;
                left: 0;
                right: 0;
                height: 5px;
                background: linear-gradient(135deg, var(--success-color) 0%, #34d399 100%);
            }
            
            .success-icon {
                width: 100px;
                height: 100px;
                background: linear-gradient(135deg, var(--success-color) 0%, #34d399 100%);
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                margin: 0 auto 30px;
                box-shadow: 0 10px 25px rgba(16, 185, 129, 0.3);
                animation: scaleIn 0.5s ease-out 0.2s both;
                position: relative;
                z-index: 2;
            }
            
            @keyframes scaleIn {
                from {
                    transform: scale(0);
                }
                to {
                    transform: scale(1);
                }
            }
            
            .success-icon i {
                font-size: 40px;
                color: white;
                animation: bounce 1s ease infinite;
            }
            
            @keyframes bounce {
                0%, 100% { transform: translateY(0); }
                50% { transform: translateY(-10px); }
            }
            
            .success-title {
                font-size: 2.2rem;
                font-weight: 800;
                color: var(--dark-color);
                margin-bottom: 15px;
                position: relative;
                z-index: 2;
            }
            
            .success-message {
                font-size: 1.1rem;
                color: #64748b;
                margin-bottom: 30px;
                line-height: 1.6;
                position: relative;
                z-index: 2;
            }
            
            /* Party Details Grid */
            .party-details-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
                gap: 20px;
                margin: 30px 0;
                position: relative;
                z-index: 2;
            }
            
            .detail-card {
                background: #f8fafc;
                border-radius: 12px;
                padding: 25px;
                border: 1px solid #e2e8f0;
                transition: var(--transition);
                text-align: left;
            }
            
            .detail-card:hover {
                transform: translateY(-5px);
                box-shadow: 0 10px 20px rgba(0, 0, 0, 0.05);
            }
            
            .detail-icon {
                width: 50px;
                height: 50px;
                background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
                border-radius: 12px;
                display: flex;
                align-items: center;
                justify-content: center;
                margin-bottom: 15px;
            }
            
            .detail-icon i {
                font-size: 20px;
                color: white;
            }
            
            .detail-content h5 {
                font-size: 0.9rem;
                font-weight: 600;
                color: #64748b;
                text-transform: uppercase;
                letter-spacing: 0.5px;
                margin-bottom: 8px;
            }
            
            .detail-content p {
                font-size: 1.1rem;
                font-weight: 500;
                color: var(--dark-color);
                margin: 0;
                word-break: break-word;
            }
            
            /* Coordinates Card */
            .coordinates-card {
                grid-column: span 2;
                background: linear-gradient(135deg, #f0f9ff 0%, #e0f2fe 100%);
                border: 1px solid #bae6fd;
            }
            
            .coordinates-grid {
                display: grid;
                grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
                gap: 15px;
                margin-top: 10px;
            }
            
            /* Action Buttons */
            .action-buttons {
                display: flex;
                gap: 15px;
                justify-content: center;
                flex-wrap: wrap;
                margin-top: 40px;
                position: relative;
                z-index: 2;
            }
            
            .btn-success-custom {
                background: linear-gradient(135deg, var(--success-color) 0%, #34d399 100%);
                border: none;
                border-radius: 12px;
                padding: 14px 30px;
                font-weight: 600;
                color: white;
                transition: var(--transition);
                box-shadow: 0 4px 15px rgba(16, 185, 129, 0.3);
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                gap: 10px;
            }
            
            .btn-success-custom:hover {
                transform: translateY(-3px);
                box-shadow: 0 6px 20px rgba(16, 185, 129, 0.4);
                color: white;
            }
            
            .btn-primary-custom {
                background: linear-gradient(135deg, var(--primary-color) 0%, var(--primary-dark) 100%);
                border: none;
                border-radius: 12px;
                padding: 14px 30px;
                font-weight: 600;
                color: white;
                transition: var(--transition);
                box-shadow: 0 4px 15px rgba(67, 97, 238, 0.3);
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                gap: 10px;
            }
            
            .btn-primary-custom:hover {
                transform: translateY(-3px);
                box-shadow: 0 6px 20px rgba(67, 97, 238, 0.4);
                color: white;
            }
            
            .btn-outline-custom {
                background: transparent;
                border: 2px solid #e2e8f0;
                border-radius: 12px;
                padding: 14px 30px;
                font-weight: 600;
                color: #64748b;
                transition: var(--transition);
                text-decoration: none;
                display: inline-flex;
                align-items: center;
                gap: 10px;
            }
            
            .btn-outline-custom:hover {
                background: #f8fafc;
                border-color: var(--primary-color);
                color: var(--primary-color);
                transform: translateY(-2px);
            }
            
            /* Confetti Animation */
            .confetti {
                position: absolute;
                width: 12px;
                height: 12px;
                border-radius: 50%;
                opacity: 0.8;
                animation: confetti-fall 3s linear forwards;
                z-index: 1;
            }
            
            @keyframes confetti-fall {
                0% {
                    transform: translateY(-100px) rotate(0deg) scale(1);
                    opacity: 1;
                }
                100% {
                    transform: translateY(600px) rotate(720deg) scale(0);
                    opacity: 0;
                }
            }
            
            /* QR Code Preview */
            .qr-preview {
                margin: 30px 0;
                padding: 20px;
                background: #f8fafc;
                border-radius: 12px;
                border: 2px dashed #cbd5e1;
                position: relative;
                z-index: 2;
            }
            
            .qr-placeholder {
                width: 150px;
                height: 150px;
                background: linear-gradient(45deg, #e2e8f0 25%, #f1f5f9 25%, #f1f5f9 50%, #e2e8f0 50%, #e2e8f0 75%, #f1f5f9 75%);
                background-size: 20px 20px;
                margin: 0 auto;
                border-radius: 8px;
                display: flex;
                align-items: center;
                justify-content: center;
                animation: shimmer 2s infinite linear;
            }
            
            @keyframes shimmer {
                0% { background-position: -40px 0; }
                100% { background-position: 40px 0; }
            }
            
            /* Responsive Design */
            @media (max-width: 768px) {
                .success-card {
                    padding: 40px 25px;
                }
                
                .success-title {
                    font-size: 1.8rem;
                }
                
                .party-details-grid {
                    grid-template-columns: 1fr;
                }
                
                .coordinates-card {
                    grid-column: span 1;
                }
                
                .action-buttons {
                    flex-direction: column;
                }
                
                .btn-success-custom,
                .btn-primary-custom,
                .btn-outline-custom {
                    width: 100%;
                    justify-content: center;
                }
            }
            
            @media (max-width: 576px) {
                body {
                    padding: 15px;
                }
                
                .success-card {
                    padding: 30px 20px;
                }
                
                .success-icon {
                    width: 80px;
                    height: 80px;
                }
                
                .success-icon i {
                    font-size: 32px;
                }
                
                .success-title {
                    font-size: 1.6rem;
                }
                
                .detail-card {
                    padding: 20px 15px;
                }
                
                .coordinates-grid {
                    grid-template-columns: 1fr;
                }
            }
            
            /* Next Steps */
            .next-steps {
                margin-top: 40px;
                padding-top: 30px;
                border-top: 1px solid #e2e8f0;
                position: relative;
                z-index: 2;
            }
            
            .next-steps h4 {
                color: var(--secondary-color);
                margin-bottom: 20px;
                font-weight: 600;
            }
            
            .step-item {
                display: flex;
                align-items: center;
                gap: 15px;
                margin-bottom: 15px;
                padding: 12px 15px;
                background: #f8fafc;
                border-radius: 10px;
                border-left: 4px solid var(--primary-color);
            }
            
            .step-number {
                width: 30px;
                height: 30px;
                background: var(--primary-color);
                color: white;
                border-radius: 50%;
                display: flex;
                align-items: center;
                justify-content: center;
                font-weight: bold;
                font-size: 0.9rem;
            }
            
            .step-content {
                flex: 1;
            }
            
            .step-content h6 {
                margin: 0;
                font-size: 0.95rem;
                color: var(--dark-color);
            }
            
            .step-content p {
                margin: 5px 0 0;
                font-size: 0.85rem;
                color: #64748b;
            }
        </style>
    </head>
    <body>
        <div class="success-container">
            <div class="success-card">
                <!-- Confetti Animation Container -->
                <div id="confetti-container"></div>
                
                <!-- Success Icon -->
                <div class="success-icon">
                    <i class="fas fa-check"></i>
                </div>
                
                <!-- Title and Message -->
                <h1 class="success-title">Party Added Successfully!</h1>
                <p class="success-message">
                    The party has been registered in the system with all details.
                    A QR code will be generated for easy scanning and access.
                </p>
                
                <!-- Party Details Grid -->
                <div class="party-details-grid">
                    <!-- Party Info -->
                    <div class="detail-card">
                        <div class="detail-icon">
                            <i class="fas fa-user-friends"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Party Number</h5>
                            <p><?php echo htmlspecialchars($party_no); ?></p>
                        </div>
                    </div>
                    
                    <!-- Party Name -->
                    <div class="detail-card">
                        <div class="detail-icon">
                            <i class="fas fa-user-tag"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Party Name</h5>
                            <p><?php echo htmlspecialchars($party_name); ?></p>
                        </div>
                    </div>
                    
                    <!-- Location -->
                    <div class="detail-card">
                        <div class="detail-icon">
                            <i class="fas fa-map-marker-alt"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Location</h5>
                            <p><?php echo htmlspecialchars($location_name); ?></p>
                        </div>
                    </div>
                    
                    <!-- Distance -->
                    <div class="detail-card">
                        <div class="detail-icon">
                            <i class="fas fa-road"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Distance</h5>
                            <p><?php echo number_format($distance, 2); ?> km</p>
                        </div>
                    </div>
                    
                    <!-- Coordinates -->
                    <div class="detail-card coordinates-card">
                        <div class="detail-icon">
                            <i class="fas fa-globe-asia"></i>
                        </div>
                        <div class="detail-content">
                            <h5>Geographical Coordinates</h5>
                            <div class="coordinates-grid">
                                <div>
                                    <h6>Longitude</h6>
                                    <p><?php echo number_format($longitude, 6); ?></p>
                                </div>
                                <div>
                                    <h6>Latitude</h6>
                                    <p><?php echo number_format($latitude, 6); ?></p>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                
                <!-- PDF Preview -->
                <div class="qr-preview">
                    <h5 class="text-center mb-3">
                        <i class="fas fa-file-pdf me-2"></i>Uploaded Document
                    </h5>
                    <div class="text-center">
                        <div class="qr-placeholder">
                            <i class="fas fa-file-pdf fa-3x" style="color: #ef4444;"></i>
                        </div>
                        <p class="mt-3 mb-2">
                            <strong><?php echo htmlspecialchars($party_no); ?>.pdf</strong>
                        </p>
                        <a href="<?php echo htmlspecialchars($upload_path); ?>" 
                           target="_blank" 
                           class="btn btn-sm btn-danger">
                            <i class="fas fa-eye me-1"></i>View PDF
                        </a>
                    </div>
                </div>
                
                <!-- Action Buttons -->
                <div class="action-buttons">
                    <a href="add_party.php" class="btn-primary-custom">
                        <i class="fas fa-plus-circle"></i>Add Another Party
                    </a>
                    
                    <a href="view_party.php?code=<?php echo urlencode($party_no); ?>" class="btn-success-custom">
                        <i class="fas fa-eye"></i>View Party Details
                    </a>
                    
                    <a href="index.php" class="btn-outline-custom">
                        <i class="fas fa-home"></i>Back to Dashboard
                    </a>
                </div>
                
                <!-- Next Steps -->
                <div class="next-steps">
                    <h4 class="text-center">What's Next?</h4>
                    <div class="step-item">
                        <div class="step-number">1</div>
                        <div class="step-content">
                            <h6>Generate QR Code</h6>
                            <p>Scan this party using the QR scanner on the dashboard</p>
                        </div>
                    </div>
                    <div class="step-item">
                        <div class="step-number">2</div>
                        <div class="step-content">
                            <h6>Share Access</h6>
                            <p>Use the party number to share access with team members</p>
                        </div>
                    </div>
                    <div class="step-item">
                        <div class="step-number">3</div>
                        <div class="step-content">
                            <h6>Track Activity</h6>
                            <p>Monitor party activity through the dashboard</p>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Bootstrap JS -->
        <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
        
        <script>
        // Create confetti animation
        function createConfetti() {
            const container = document.getElementById('confetti-container');
            const colors = [
                '#10b981', '#34d399', '#22c55e', '#16a34a', // Greens
                '#4361ee', '#3a56d4', '#3a0ca3', '#4cc9f0', // Blues
                '#8b5cf6', '#7c3aed', '#6d28d9', '#a78bfa'  // Purples
            ];
            
            for (let i = 0; i < 80; i++) {
                const confetti = document.createElement('div');
                confetti.className = 'confetti';
                
                // Random properties
                const size = Math.random() * 15 + 5;
                const left = Math.random() * 100;
                const color = colors[Math.floor(Math.random() * colors.length)];
                const duration = Math.random() * 2 + 2;
                const delay = Math.random() * 3;
                const shape = Math.random() > 0.5 ? '50%' : '0';
                
                // Apply styles
                confetti.style.width = `${size}px`;
                confetti.style.height = `${size}px`;
                confetti.style.backgroundColor = color;
                confetti.style.left = `${left}%`;
                confetti.style.borderRadius = shape;
                confetti.style.animationDuration = `${duration}s`;
                confetti.style.animationDelay = `${delay}s`;
                
                container.appendChild(confetti);
                
                // Remove confetti after animation completes
                setTimeout(() => {
                    if (confetti.parentNode) {
                        confetti.remove();
                    }
                }, (duration + delay) * 1000);
            }
        }
        
        // Create confetti on page load
        document.addEventListener('DOMContentLoaded', () => {
            // Initial confetti burst
            createConfetti();
            
            // Create more confetti every 5 seconds
            setInterval(() => {
                createConfetti();
            }, 5000);
            
            // Auto-redirect after 60 seconds (optional)
            setTimeout(() => {
                const redirect = confirm('Would you like to return to the dashboard?');
                if (redirect) {
                    window.location.href = 'index.php';
                }
            }, 60000);
            
            // Add some interactive effects
            const buttons = document.querySelectorAll('.btn-success-custom, .btn-primary-custom, .btn-outline-custom');
            buttons.forEach(button => {
                button.addEventListener('mouseenter', () => {
                    createConfetti();
                });
            });
            
            // Success icon click animation
            const successIcon = document.querySelector('.success-icon');
            successIcon.addEventListener('click', () => {
                successIcon.style.animation = 'scaleIn 0.5s ease-out';
                setTimeout(() => {
                    successIcon.style.animation = '';
                }, 500);
                createConfetti();
            });
        });
        </script>
    </body>
    </html>
    <?php
}
?>