<?php
// set_page_password.php - Admin page to set page passwords
session_start();
include 'db_connect.php';
include 'navbar.php';

// Check if user is admin (you can modify this based on your user roles)
if (!isset($_SESSION['user_id']) || $_SESSION['role'] != 'admin') {
    header('Location: login.php');
    exit;
}

$message = '';
$message_type = '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    $page_name = mysqli_real_escape_string($conn, $_POST['page_name']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    
    if ($password !== $confirm_password) {
        $message = "Passwords do not match!";
        $message_type = "danger";
    } else {
        // Hash the password
        $hashed_password = password_hash($password, PASSWORD_DEFAULT);
        
        // Check if entry exists
        $check_query = "SELECT id FROM page_passwords WHERE page_name = ?";
        $check_stmt = $conn->prepare($check_query);
        $check_stmt->bind_param("s", $page_name);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        
        if ($check_result->num_rows > 0) {
            // Update existing
            $update_query = "UPDATE page_passwords SET page_password = ? WHERE page_name = ?";
            $update_stmt = $conn->prepare($update_query);
            $update_stmt->bind_param("ss", $hashed_password, $page_name);
            
            if ($update_stmt->execute()) {
                $message = "Password updated successfully!";
                $message_type = "success";
            } else {
                $message = "Error updating password: " . $conn->error;
                $message_type = "danger";
            }
            $update_stmt->close();
        } else {
            // Insert new
            $insert_query = "INSERT INTO page_passwords (page_name, page_password) VALUES (?, ?)";
            $insert_stmt = $conn->prepare($insert_query);
            $insert_stmt->bind_param("ss", $page_name, $hashed_password);
            
            if ($insert_stmt->execute()) {
                $message = "Password set successfully!";
                $message_type = "success";
            } else {
                $message = "Error setting password: " . $conn->error;
                $message_type = "danger";
            }
            $insert_stmt->close();
        }
        
        $check_stmt->close();
    }
}

// Get existing passwords
$passwords_query = "SELECT * FROM page_passwords ORDER BY page_name";
$passwords_result = $conn->query($passwords_query);
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Set Page Password - Admin</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .password-card {
            border-radius: 20px;
            box-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
            border: none;
            margin-bottom: 30px;
        }
        
        .password-card .card-header {
            background: linear-gradient(135deg, #4361ee, #3a0ca3);
            color: white;
            border-radius: 20px 20px 0 0;
            padding: 1.5rem;
            border: none;
        }
        
        .password-card .card-body {
            padding: 2rem;
        }
        
        .form-control, .form-select {
            border-radius: 10px;
            padding: 0.75rem 1rem;
            border: 2px solid #e0e7ff;
        }
        
        .form-control:focus, .form-select:focus {
            border-color: #4361ee;
            box-shadow: 0 0 0 0.25rem rgba(67, 97, 238, 0.15);
        }
        
        .btn-primary {
            background: linear-gradient(135deg, #4361ee, #3a0ca3);
            border: none;
            border-radius: 10px;
            padding: 0.75rem 2rem;
            font-weight: 600;
        }
        
        .btn-primary:hover {
            transform: translateY(-2px);
            box-shadow: 0 5px 15px rgba(67, 97, 238, 0.4);
        }
        
        .table-custom {
            border-radius: 15px;
            overflow: hidden;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        }
        
        .table-custom thead {
            background: linear-gradient(135deg, #4361ee, #3a0ca3);
            color: white;
        }
        
        .table-custom th {
            font-weight: 600;
            padding: 1rem;
        }
        
        .table-custom td {
            padding: 1rem;
            vertical-align: middle;
        }
        
        .badge-password {
            background: linear-gradient(135deg, #10b981, #059669);
            color: white;
            padding: 0.5rem 1rem;
            border-radius: 50px;
            font-weight: 500;
        }
    </style>
</head>
<body>
    <div class="container mt-5">
        <div class="row justify-content-center">
            <div class="col-md-8">
                <div class="card password-card">
                    <div class="card-header">
                        <h3 class="mb-0"><i class="fas fa-key me-2"></i>Set Page Password</h3>
                        <p class="mb-0 mt-2">Configure password protection for specific pages</p>
                    </div>
                    <div class="card-body">
                        <?php if ($message): ?>
                            <div class="alert alert-<?php echo $message_type; ?> alert-dismissible fade show" role="alert">
                                <i class="fas <?php echo $message_type == 'success' ? 'fa-check-circle' : 'fa-exclamation-circle'; ?> me-2"></i>
                                <?php echo $message; ?>
                                <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                            </div>
                        <?php endif; ?>
                        
                        <form method="POST" action="" id="passwordForm">
                            <div class="mb-4">
                                <label for="page_name" class="form-label fw-bold">
                                    <i class="fas fa-file me-2"></i>Select Page
                                </label>
                                <select class="form-select" id="page_name" name="page_name" required>
                                    <option value="">-- Choose Page --</option>
                                    <option value="add_party.php">Add Party Page</option>
                                    <option value="edit_party.php">Edit Party Page</option>
                                    <option value="view_parties.php">View Parties Page</option>
                                    <option value="reports.php">Reports Page</option>
                                    <option value="settings.php">Settings Page</option>
                                </select>
                            </div>
                            
                            <div class="mb-4">
                                <label for="password" class="form-label fw-bold">
                                    <i class="fas fa-lock me-2"></i>Password
                                </label>
                                <div class="input-group">
                                    <input type="password" class="form-control" id="password" name="password" 
                                           placeholder="Enter page password" required minlength="6">
                                    <button class="btn btn-outline-secondary" type="button" id="togglePassword">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                                <div class="form-text">Minimum 6 characters</div>
                            </div>
                            
                            <div class="mb-4">
                                <label for="confirm_password" class="form-label fw-bold">
                                    <i class="fas fa-lock me-2"></i>Confirm Password
                                </label>
                                <div class="input-group">
                                    <input type="password" class="form-control" id="confirm_password" 
                                           name="confirm_password" placeholder="Confirm password" required minlength="6">
                                    <button class="btn btn-outline-secondary" type="button" id="toggleConfirmPassword">
                                        <i class="fas fa-eye"></i>
                                    </button>
                                </div>
                            </div>
                            
                            <div class="text-center">
                                <button type="submit" class="btn btn-primary">
                                    <i class="fas fa-save me-2"></i>Save Password
                                </button>
                                <a href="index.php" class="btn btn-outline-secondary ms-2">
                                    <i class="fas fa-times me-2"></i>Cancel
                                </a>
                            </div>
                        </form>
                    </div>
                </div>
                
                <!-- Existing Passwords Table -->
                <?php if ($passwords_result->num_rows > 0): ?>
                <div class="card password-card mt-4">
                    <div class="card-header">
                        <h4 class="mb-0"><i class="fas fa-list me-2"></i>Existing Page Passwords</h4>
                    </div>
                    <div class="card-body">
                        <div class="table-responsive">
                            <table class="table table-custom">
                                <thead>
                                    <tr>
                                        <th>Page Name</th>
                                        <th>Status</th>
                                        <th>Last Updated</th>
                                        <th>Action</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($row = $passwords_result->fetch_assoc()): ?>
                                    <tr>
                                        <td><i class="fas fa-file me-2 text-primary"></i><?php echo $row['page_name']; ?></td>
                                        <td><span class="badge-password"><i class="fas fa-check-circle me-1"></i>Protected</span></td>
                                        <td><?php echo date('d M Y H:i', strtotime($row['updated_at'])); ?></td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-primary edit-password" 
                                                    data-page="<?php echo $row['page_name']; ?>">
                                                <i class="fas fa-edit"></i> Change
                                            </button>
                                        </td>
                                    </tr>
                                    <?php endwhile; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Toggle password visibility
        document.getElementById('togglePassword').addEventListener('click', function() {
            const passwordInput = document.getElementById('password');
            const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
            passwordInput.setAttribute('type', type);
            this.querySelector('i').classList.toggle('fa-eye');
            this.querySelector('i').classList.toggle('fa-eye-slash');
        });
        
        document.getElementById('toggleConfirmPassword').addEventListener('click', function() {
            const passwordInput = document.getElementById('confirm_password');
            const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
            passwordInput.setAttribute('type', type);
            this.querySelector('i').classList.toggle('fa-eye');
            this.querySelector('i').classList.toggle('fa-eye-slash');
        });
        
        // Form validation
        document.getElementById('passwordForm').addEventListener('submit', function(e) {
            const password = document.getElementById('password').value;
            const confirm = document.getElementById('confirm_password').value;
            
            if (password !== confirm) {
                e.preventDefault();
                alert('Passwords do not match!');
            }
        });
        
        // Edit password buttons
        document.querySelectorAll('.edit-password').forEach(button => {
            button.addEventListener('click', function() {
                const page = this.dataset.page;
                document.getElementById('page_name').value = page;
                document.getElementById('password').focus();
            });
        });
    </script>
</body>
</html>
<?php $conn->close(); ?>